#!/usr/bin/perl

use Inline "C";

my $str1 = "hello ";
my $str2 = "goodbye";

print "Stack based:           " . strconcat_stack($str1, $str2) ."\n";
print "Native Perl API:       " . strconcat_api($str1, $str2) ."\n";
print "Use temporary SV:      " . strconcat_tempsv($str1, $str2) ."\n";
print "Explicit New/Safefree: " . strconcat_mallocfree($str1, $str2) ."\n";
print "Mortal allocator:      " . strconcat_mortalbuffer($str1, $str2) ."\n";

__END__
__C__

/* Static helper routines */

/* We use this routine so that we do not have to repeat ourselves */
static STRLEN mystrconcat (char* str1, char* str2, char* outstr) {
   strcpy( outstr, (const char*)str1 );
   strcat( outstr, (const char*)str2 );
   return strlen( outstr );
}

/* Mortal memory allocator */
static void * get_mortalspace ( size_t nbytes ) {
  SV * mortal;
  mortal = sv_2mortal( NEWSV(0, nbytes ) );
  return (void *)SvPVX(mortal);
}

/* difference Inline implementations of strconcat */

/* cat using Inline stack manipulation directly */
void strconcat_stack( char * str1, char * str2 ) {
  SV* outsv;
  Inline_Stack_Vars;
  int len = strlen(str1) + strlen(str2) +1;
  outsv = sv_2mortal(NEWSV(0, len));
  len = mystrconcat(str1, str2, SvPVX(outsv));
  SvPOK_on(outsv);
  SvCUR_set(outsv, len);
  Inline_Stack_Reset;
  Inline_Stack_Push(outsv);
  Inline_Stack_Done;
}

/* Allocate an SV and use its buffer directly */
SV * strconcat_tempsv( char * str1, char * str2 ) {
  SV* outsv;
  int len = strlen(str1) + strlen(str2) +1;

  outsv = NEWSV(0, len);

  /* external routine that does the concatenation */
  len = mystrconcat(str1, str2, SvPVX(outsv));
  SvPOK_on(outsv);
  /* Page 241 has len-1. This is incorrect */
  SvCUR_set(outsv, len);
  return outsv;
}

SV * strconcat_mallocfree( char * str1, char * str2 ) {
  char * outstr;
  SV* outsv;
  int len = strlen(str1) + strlen(str2) +1;
  New(0, outstr, len, char);
  /* Call to private concatenation function */
  mystrconcat(str1, str2, outstr);
  outsv = newSVpv(outstr, len);
  Safefree(outstr);
  return outsv;
}

char * strconcat_mortalbuffer( char * str1, char * str2 ) {
  char* outstr;
  int len = strlen(str1) + strlen(str2) +1;
  outstr = (char *)get_mortalspace( len );
  mystrconcat(str1, str2, outstr);
  return outstr;
}

/* Use the perl API directly ! */
SV * strconcat_api ( char * str1, char * str2 ) {
  SV * outsv;
  outsv = newSVpv( str1, 0 );
  sv_catpv( outsv, str2 );
  return outsv;
}
