
#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

typedef int intArray;

/* This typedef is for T_OPAQUEARRAY so that we can use
   it in the same file as our T_ARRAY demo */
typedef int intPacked;

/* This is the generic mortal allocator */

void * get_mortalspace ( int nbytes ) {
  SV * mortal;
  mortal = sv_2mortal( NEWSV(2040, nbytes ) );
  return (void *) SvPVX( mortal );
}


/* Memory allocator - if you use this allocator [by renaming it as 
   intArrayPtr] (which uses New) you must also use Safefree in CLEANUP */
intArray * intArrayPtr_NewFree ( int num ) {
  intArray * array;

  /* Need to allocate memory for the integer array of the 
     specified size */
  New(0,array, num, intArray );
  return array;
}

/* Memory allocator - use mortal SV  so no need for CLEANUP */
intArray * intArrayPtr ( int num ) {
  intArray * array;
  SV * mortal;
  mortal = sv_2mortal( NEWSV(2040, num * sizeof(intArray) ) );
  array = (intArray *) SvPVX( mortal );
  return array;
}

/* Add up everything in an int array */
/* Args: the number of things to add, pointer to array */

int sum ( int num, intArray * array ) {
  int thesum = 0;
  int count;
  for (count = 0; count < num; count++) {
    thesum += array[count];
  }
  return thesum;
}

MODULE = Arrays		PACKAGE = Arrays


# Using the (updated) T_ARRAY typemap

int
sum_as_list( array, ...)
  intArray * array
 CODE:
  /* ix_array is the total number of elements */
  RETVAL = sum( ix_array, array );
 OUTPUT:
  RETVAL


# Using an array reference as input

int
sum_as_ref( avref )
  AV * avref;
 PREINIT:
  int len;
  int i;
  SV ** elem;
  intArray * array;
 CODE:
  len = av_len( avref ) + 1;
  array = intArrayPtr( len );

  /* copy numbers from array */
  for (i=0; i<len; i++) {
    elem = av_fetch( avref, i, 0);
    if (elem == NULL) {
      array[i] = 0;
    } else {
      array[i] = SvIV( *elem );
    }
  }
  RETVAL = sum( len, array );
 OUTPUT:
  RETVAL 

# Using a packed string
#  Call with  sum_t_packed( pack("i*",@arr) );
# Pass in an SV* so that we can determine the length of the
# packed string

int
sum_as_packed( packed )
  SV * packed
 PREINIT:
  int len;
  intArray * array;
 CODE:
  array = (intArray *)SvPV_nolen( packed );
  len = SvCUR( packed ) / sizeof(intArray);
  RETVAL = sum( len, array );
 OUTPUT:
  RETVAL

int
sum_as_packed2( len, packed )
  int len
  char * packed
 CODE:
  RETVAL = sum( len, (intArray *)packed );
 OUTPUT:
  RETVAL

# These are for returning arrays as contrived examples

intArray *
test_t_array()
 PREINIT:
  intArray test[2];
  U32 size_RETVAL;
 CODE:
  test[0] = 1; test[1] = 2;
  size_RETVAL = 2;
  RETVAL = test;
 OUTPUT:
  RETVAL
 CLEANUP:
  XSRETURN(size_RETVAL);

array(int, 3)
return_packed()
 PREINIT:
  intArray test[3];
 CODE:
  test[0] = 1; test[1] = 2; test[2] = 3;
  RETVAL = test;
 OUTPUT:
  RETVAL

intPacked *
return_npacked()
 PREINIT:
  U32 size_RETVAL;
  intPacked test[3];
 CODE:
  test[0] = 1; test[1] = 2; test[2] = 3;
  size_RETVAL = 3;
  RETVAL = test;
 OUTPUT:
  RETVAL
